/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.BuildFileRule;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

/**
 */
public class FailTest {

    @Rule
    public final BuildFileRule buildRule = new BuildFileRule();

    @Before
    public void setUp() {
        buildRule.configureProject("src/etc/testcases/taskdefs/fail.xml");
    }

    @Test
    public void test1() {
        try {
            buildRule.executeTarget("test1");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("No message", ex.getMessage());
        }
    }

    @Test
    public void test2() {
        try {
            buildRule.executeTarget("test2");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("test2", ex.getMessage());
        }
    }

    @Test
    public void testText() {
        try {
            buildRule.executeTarget("testText");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("testText", ex.getMessage());
        }
    }

    @Test
    public void testIf() {
        buildRule.executeTarget("testIf");
        buildRule.getProject().setProperty("foo", "");
        try {
            buildRule.executeTarget("testIf");
            fail("testIf must fail if foo has been set") ;
        } catch (BuildException ex) {
            //TODO assert result
        }
    }

    @Test
    public void testUnless() {
        try {
            buildRule.executeTarget("testUnless");
            fail("testUnless must fail unless foo has been set") ;
        } catch (BuildException ex) {
            //TODO assert rules
        }
        buildRule.getProject().setProperty("foo", "");
        buildRule.executeTarget("testUnless");

    }

    /**
     * see that the different combinations work, and
     * that the autogenerated text contains information
     * about which condition was not met
     */
    @Test
    public void testIfAndUnless() {
        //neither
        buildRule.executeTarget("testIfAndUnless");
        buildRule.getProject().setProperty("if", "");
        try {
            buildRule.executeTarget("testIfAndUnless");
            fail("expect fail on defined(if)") ;
        } catch (BuildException ex) {
            assertEquals("if=if and unless=unless", ex.getMessage());
        }
        buildRule.getProject().setProperty("unless", "");
        //this call should succeed as unless overrides if
        buildRule.executeTarget("testIfAndUnless");
    }
    /**
     * see that the different combinations work, and
     * that the autogenerated text contains information
     * about which condition was not met
     */
    @Test
    public void testIfAndUnless2() {
        buildRule.getProject().setProperty("unless", "");
        buildRule.executeTarget("testIfAndUnless");
    }

    @Test
    public void testNested1() {
        try {
            buildRule.executeTarget("testNested1");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("condition satisfied", ex.getMessage());
        }
    }

    @Test
    public void testNested2() {
        buildRule.executeTarget("testNested2");
    }

    @Test
    public void testNested3() {
        try {
            buildRule.executeTarget("testNested3");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("testNested3", ex.getMessage());
        }
    }

    @Test
    public void testNested4() {
        String specificMessage = "Nested conditions "
          + "not permitted in conjunction with if/unless attributes";

        char[] c = {'a', 'b', 'c'};
        StringBuffer target = new StringBuffer("testNested4x");

        for (int i = 0; i < c.length; i++) {
            target.setCharAt(target.length() - 1, c[i]);
            try {
                buildRule.executeTarget(target.toString());
                fail("it is required to fail :-)") ;
            } catch (BuildException ex) {
                assertEquals(specificMessage, ex.getMessage());
            }
        }
    }

    @Test
    public void testNested5() {
        try {
            buildRule.executeTarget("testNested5");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("Only one nested condition is allowed.", ex.getMessage());
        }
    }

    @Test
    public void testNested6() {
        try {
            buildRule.executeTarget("testNested6");
            fail("it is required to fail :-)") ;
        } catch (BuildException ex) {
            assertEquals("testNested6\ntestNested6\ntestNested6", ex.getMessage());
        }
    }

    @Test
    public void testNested7() {
        String specificMessage = "A single nested condition is required.";

        char[] c = {'a', 'b'};
        StringBuffer target = new StringBuffer("testNested7x");

        for (int i = 0; i < c.length; i++) {
            target.setCharAt(target.length() - 1, c[i]);
            try {
                buildRule.executeTarget(target.toString());
                fail("it is required to fail :-)") ;
            } catch (BuildException ex) {
                assertEquals(specificMessage, ex.getMessage());
            }
        }
    }

}
