// ---------------------------------------------------------------------------
// - Number.cpp                                                              -
// - standard object library - number class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Number.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // return a hexadecimal representation of this number

  String Number::tohexa (void) const {
    throw Exception ("number-error", 
		     "cannot generate a hexadecimal representation", repr ());
  }

  // return a hexadecimal string of this number

  String Number::tohstr (void) const {
    throw Exception ("number-error", 
		     "cannot generate a hexadecimal string", repr ());
  }

  // return a formatted string of this number with a precision

  String Number::format (const long psize) const {
    if (psize < 0) {
      throw Exception ("number-error", "negative precision with format");
    }
    rdlock ();
    try {
      // get the base number
      String result = tostring ();
      // check for additional precision
      if (psize > 0) {
	result += '.';
	for (long i = 0; i < psize; i++) result += '0';
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------
  
  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_FORMAT = zone.intern ("format");
  static const long QUARK_TOHEXA = zone.intern ("to-hexa");
  static const long QUARK_TOHSTR = zone.intern ("to-hexa-string");

  // return true if the given quark is defined

  bool Number::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Literal::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* Number::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_FORMAT) return new String (format (0));
      if (quark == QUARK_TOHEXA) return new String (tohexa ());
      if (quark == QUARK_TOHSTR) return new String (tohstr ());
    }
    
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_FORMAT) {
	long precision = argv->getlong (0);
	return new String (format (precision));
      }
    }

    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
