# -*- coding: utf-8; indent-tabs-mode: t; tab-width: 4 -*-

#  Copyright (C) 2008-2010 by Canonical Ltd
#  Copyright (C) 2012 Koichi Akabe <vbkaisetsu@gmail.com>
# 
#  gLogic is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the
#  Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
# 
#  gLogic is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#  See the GNU General Public License for more details.
# 
#  You should have received a copy of the GNU General Public License along
#  with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import twitter
import bzrlib
import bzrlib.api

from bzrlib import (
	config as globalconfig,
	hooks,
	)

from info import (
	bzr_plugin_version as version_info,
	bzr_compatible_versions,
	)

from i18n import (
	gettext as _,
	install as i18n_install,
	)

from keys import *

if version_info[3] == 'final':
	version_string = '%d.%d.%d' % version_info[:3]
else:
	version_string = '%d.%d.%d%s%d' % version_info
__version__ = version_string

bzrlib.api.require_any_api(bzrlib, bzr_compatible_versions)

from bzrlib.commands import plugin_cmds

plugin_cmds.register_lazy('cmd_tweet_auth', ["twitter-auth"], 'bzrlib.plugins.tweet.cmds')
plugin_cmds.register_lazy('cmd_tweet_branchid', ["twitter-branchid"], 'bzrlib.plugins.tweet.cmds')

TWEET_MAXLEN = 140

i18n_install()

service_datas = [
	("twitter", None, twitter_consumer_key, twitter_consumer_sec),
	#("identica", "https://identi.ca/api/", identica_consumer_key, identica_consumer_sec)
]


def get_api(service_name, base_url, consumer_key, consumer_sec):

	gs = globalconfig.GlobalStack()

	token_key = gs.get("%s_token_key" % service_name)
	token_secret = gs.get("%s_token_secret" % service_name)
	if token_key is None or token_secret is None:
		return None
	return twitter.Api(base_url=base_url, consumer_key=consumer_key, consumer_secret=consumer_sec,
	                  access_token_key=token_key, access_token_secret=token_secret)


def post_push_hook(push_result):

	old_rev = push_result.old_revno
	new_rev = push_result.new_revno

	if old_rev == new_rev:
		return

	branch = push_result.source_branch
	config = branch.get_config()
	branch_name = config.get_user_option("twitter_branch_id")

	if branch_name is None:
		return


	for service_data in service_datas:

		api = get_api(service_data[0], service_data[1], service_data[2], service_data[3])
		if api is None:
			continue

		tweet_on_push = config.get_user_option("tweet_on_push")
		if tweet_on_push is None:
			gs = globalconfig.GlobalStack()
			tweet_on_push = gs.get("tweet_on_push")

		if tweet_on_push == "yes":
			for revno in range(old_rev + 1, new_rev + 1):
				revid = branch.dotted_revno_to_revision_id((revno,))
				revision = branch.repository.get_revision(revid)
				tweet_revision(service_data[0], api, revision, branch_name, revno)
	
		tweetstr = _("bzr push: %(name)s was pushed up from %(old)d to %(new)d [bzr-tweet]") % {"name": branch_name, "old": old_rev, "new": new_rev}

		try:
			api.PostUpdate(tweetstr)
		except Exception as e:
			print("Error: %s" % e)
		else:
			print((_("Tweet on %s: %s") % (service_data[0], tweetstr)).encode("utf-8"))


def post_commit_hook(local, master, old_revno, old_revid, new_revno, new_revid):

	branch = local or master
	config = branch.get_config()
	branch_name = config.get_user_option("twitter_branch_id")
	if branch_name is None:
		return
	tweet_on_push = config.get_user_option("tweet_on_push")
	if tweet_on_push == "yes":
		return
	
	for service_data in service_datas:
		api = get_api(service_data[0], service_data[1], service_data[2], service_data[3])
		if api is None:
			continue

		revision = branch.repository.get_revision(new_revid)
		tweet_revision(service_data[0], api, revision, branch_name, new_revno)


def tweet_revision(service_name, api, revision, branch_name, revno):

	message = revision.message.splitlines()[0]

	tweet_fmtstr = _("bzr commit: %(name)s has the new revid: %(revid)d: %(msg)s [bzr-tweet]")
	msgmax = TWEET_MAXLEN - len(tweet_fmtstr % {"name": branch_name, "revid": revno, "msg": ""})

	if(msgmax < len(message)):
		message = message[:msgmax - 3] + "..."

	tweetstr = tweet_fmtstr % {"name": branch_name, "revid": revno, "msg": message}

	try:
		api.PostUpdate(tweetstr)
	except Exception as e:
		print("Error: %s" % e)
	else:
		print((_("Tweet on %s: %s") % (service_name, tweetstr)).encode("utf-8"))


# install hooks
hooks.install_lazy_named_hook("bzrlib.branch", "Branch.hooks", "post_push", post_push_hook, "push tweet hook")
hooks.install_lazy_named_hook("bzrlib.branch", "Branch.hooks", "post_commit", post_commit_hook, "commit tweet hook")
