#!/usr/bin/perl -w
#==============================================================================
#
#    Automated Task Processing, Publishing, and SVN Update Script
#    Copyright (C) 2000-2012 by Eric Sunshine <sunshine@sunshineco.com>
#                       2006 by Marten Svanfeldt <developer@svanfeldt.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
#==============================================================================
#------------------------------------------------------------------------------
# jobber-svn.pl
#
# A generalized tool for automatically performing maintenance tasks based upon
# the contents of a project's SVN repository.  Tasks and configuration details
# are specified via an external configuration file. The overall operation of
# jobber-svn.pl follows these steps:
#
# (1) Extract project tree from SVN repository.
# (2) Perform a set of tasks upon the extracted project tree.
# (3) Commit changed files back to the repository.
# (4) Optionally publish project documentation for online browsing and
#     download.
#
# Some common tasks for which jobber-svn.pl may be called upon include:
#
# * Reparing outdated resources which can be generated automatically from
#   other project content.
# * Generating project files for various build tools.
# * Generating documentation in various formats from input files, such as
#   Texinfo or header files.
#
# This script takes special care to invoke the appropriate SVN commands to add
# new files and directories to the repository, and to remove obsolete files.
#
# Typically this script is run via the 'cron' daemon.  See the cron(8),
# crontab(1), and crontab(8) man pages for information specific to cron.  A
# typical crontab entry which runs this script at 02:13 each morning might look
# like this:
#
#     MAILTO = sunshine@sunshineco.com
#     13 2 * * * $HOME/bin/jobber-svn.pl
#
# The script makes no attempt to perform any sort of SVN authentication.  It is
# the client's responsibility to authenticate with the SVN server if necessary.
# For access the easiest way to do so is to login to the SVN server  one time 
# manually using the appropriate identity.  Once logged in successfully, 
# the authentication information is stored in $(HOME)/.subversion/ directory
# and remains there.  From that point onward, SVN considers the account as
# having been authenticated.
# The identity used for SVN access must have "write" access to the repository 
# if files changed by the performed tasks are to be committed back to the 
# repository.
#
#------------------------------------------------------------------------------
#
# At run-time, jobber-svn.pl relies upon an external configuration file which
# specifies the list of tasks to be performed, for settings controlling access
# to the SVN repository, and for indicating where documentation should be
# published.  The configuration file may be specified via --config command-line
# option (i.e. "--config myjobber.cfg"). If it is not specified via --config,
# then jobber-svn.pl looks for a file named jobber.cfg or .jobber, first in the
# current working directory, and then in the home directory ($HOME). If no
# configuration file is found, the script aborts.
#
# At the command-line, jobber-svn.pl allows users to define arbitrary run-time
# properties via the --set option. These properties can be accessed within the
# configuration file by consulting the %jobber_properties hash. For instance,
# the command-line argument "--set foo=bar" sets "bar" as the value of
# $jobber_properties{foo}.
#
# The following Perl variables can be set in the configuration file:
#
# $jobber_project_root [required]
#     Root directory of the project.  This is the top-level directory created
#     as a side-effect of retrieving the files from SVN. No default.
#
# $jobber_svn_base_url [required]
#     The URL used as base url when invoking SVN commands. The specified value
#     must allow "write" access to the repository if files are to be commited
#     back to the repository. No default.
#
# $jobber_svn_user [required]
#     The user to use for authentication when doing write operations on the
#     repository. Notice that the password/certificate must be configured 
#     as noted above.
#
# $jobber_svn_flags [optional]
#     Additional flags to pass to each of the `svn' command invocations.  An
#     obvious example would be to set this variable to "-z9" to enable
#     compression. No default.
#
# $jobber_browseable_dir [conditional]
#     Absolute path of directory into which generated documentation should be
#     copied for online browsing. This setting is required if any tasks publish
#     documentation, otherwise it is optional. No default.
#
# $jobber_package_dir [conditional]
#     Absolute path of directory into which archives of generated documentation
#     are placed to make them available for download in package form.  This
#     setting is required if any tasks publish documentation, otherwise it is
#     optional.  No default.
#
# $jobber_public_group [optional]
#     Group name to which to assign ownership of all directories which will
#     exist after script's termination (such as the "browseable" directory).
#     May be 'undef' if no special group should be assigned. Default: undef
#
# $jobber_public_mode [optional]
#     Mode to which to assign all directories which will exist after script's
#     termination (such as the "browseable" directory).  Use this in
#     conjunction with $jobber_public_group to make directories group-writable,
#     for example. For this purpose, set it to the octal value 0775.  May be
#     'undef' if no special mode should be assigned. Default: undef
#
# $jobber_temp_dir [optional]
#     Absolute path of temporary working directory where all processing should
#     occur.  The script cleans up after itself, so nothing will be left in
#     this directory after the script terminates. Default: "/tmp"
#
# @jobber_tasks [required]
#     A list of tasks to perform on the checked-out source tree.  Typical tasks
#     are those which repair outdated files, and those which generate
#     human-consumable documentation from various sources.  Files generated or
#     repaired by the tasks can then optionally be committed back to the SVN
#     repository and/or published for browsing or download. Each task's
#     "command" is invoked in the top-level directory of the project tree
#     ($jobber_project_root).
#
#     Many projects need to be "configured" before various tasks can be
#     performed (often by running some sort of configuration script). If this
#     true for your project, then your very first task should invoke the
#     command(s) which configure the project.
#
#     Each task record is a dictionary which contains the following keys:
#
#     name [required]
#         Human-readable name for this task; used in status messages.
#     action [required]
#         Human-readable verb which describes the action performed by this
#         target. It is combined with the value of the "name" key to construct
#         an informative diagnositc message.
#     command [optional]
#         The actual command which is invoked to perform this task. It may
#         repair outdated files or generate a set of files (such as HTML
#         documentation).
#     newdirs [optional]
#         Refers to an array of directories into which files are generated by
#         this task.  This key should only be present if new files are created
#         by this target.
#     olddirs [optional]
#         Refers to an array of existing directories where files generated by
#         this task are stored in the SVN repository.  If the "newdirs" key is
#         omitted, then the directories mentioned by "olddirs" are those
#         containing files modified in-place by the command, rather than
#         generated anew in a different location.  If both "newdirs" and
#         "olddirs" are present, then entries in "newdirs" must correspond to
#         entries in "olddirs", and each directory in "newdirs" must exactly
#         mirror the layout and hierarchy of each corresponding directory in
#         "olddirs".
#     log [optional]
#         Log message to use for SVN transactions involving this target.  The
#         keys "olddirs" and "log" should be present only if the files
#         generated by this target should be committed back into the SVN
#         repository.
#     export [optional]
#         Refers to a sub-dictionary which describes how to export the target.
#         This key should be present only if the files generated by the task
#         should be published for browsing and downloading.  If this key is
#         used, then one or both of "olddirs" and "newdirs" must also be
#         present.  The sub-dictionary referenced by the "export" entry may
#         contain the following keys:
#
#         dir [required]
#             Directory name into which files for this task are published.
#             Online browseable files are placed into
#             $jobber_browseable_dir/$dir, and downloadable packages are placed
#             into $jobber_package_dir/$dir.
#         package-dir [optional]
#             Directory name into which downloadable package files for this
#             task are published. If omitted, "dir" is used.
#         name [required]
#             Base package name used when generating downloadable packages via
#             @jobber_archivers (see below).  When published, the base package
#             name is combined with the archiver's file extension and placed
#             within the appropriate subdirectory of $jobber_package_dir.
#             *NOTE* Presently, the implementation is limited to only exporting
#             the first directory referenced by the sibling "newdirs" key.
#         appear [optional]
#             Controls the appearance of the directory in the generated
#             package.  For example, when packaging files from a temporary
#             build directory named "./out/docs/html/manual", it might be
#             preferable if it actually appeared as "CS/docs/html/manual" in
#             the downloadable package.
#         browseable-postprocess [optional]
#             Allows specification of a post-processing step for documentation
#             which is being made available for online browsing.  The value of
#             this key is any valid shell command.  It is invoked after the
#             files have been copied to the browseable directory. If the
#             meta-token ~T appears in the command, then the path of the
#             directory into which the files have been published is
#             interpolated into the command in its place.
#
# @jobber_archivers [optional]
#     A list of archiver records.  An archiver is used to generate a package
#     from an input directory.  Each entry in this list is a dictionary which
#     contains the following keys:
#
#     name [required]
#         Specifies the archiver's printable name.
#     extension [required]
#         File extension for the generated archive file.
#     command [required]
#         Command template which describes how to generate the given archive.
#         The template may contain the meta-token ~S and ~D.  The name of the
#         source directory is interpolated into the command in place of ~S, and
#         the destination package name is interpolated in place of ~D.
#
#     As a convenience, jobber-svn.pl defines several pre-fabricated archiver
#     dictionaries:
#
#     $ARCHIVER_BZIP2
#         Archives with 'tar' and compresses with 'bzip2'. Extension: .tar.bz2
#     $ARCHIVER_GZIP
#         Archives with 'tar' and compresses with 'gzip'. Extension: .tgz
#     $ARCHIVER_ZIP
#         Archives and compresses with 'zip'. Extension: .zip
#     $ARCHIVER_LZMA
#         Archives with 'tar' and compresses with 'lzma'. Extension: .tar.lzma
#
#     Default: ($ARCHIVER_BZIP2, $ARCHIVER_GZIP, $ARCHIVER_ZIP)
#
#------------------------------------------------------------------------------
#
# To-Do List
#
# * Generalize into a "job" processing mechanism.  Each job could reside within
#   its own source file.  Jobs such as checking out files from SVN, committing
#   changes to SVN, and publishing browseable and downloadable documentation
#   can perhaps just be additional tasks in the @jobber_tasks array.
# * The mechanism for publishing packages for download and online browsing
#   needs to be generalized further.  It is still somewhat geared toward the
#   publication of documentation packages and is, thus, not flexible enough to
#   publish packages which do not follow the directory structure designed for
#   documentation.
# * Eliminate the restriction in which only the first directory listed by the
#   "newdirs" array is exported by the "exports" key.
#
#------------------------------------------------------------------------------
use Carp;
use File::Basename;
use File::Copy;
use File::Find;
use File::Path;
use FileHandle;
use Getopt::Long;
use Cwd;
use strict;
use warnings;
$Getopt::Long::ignorecase = 0;

my $PROG_NAME = 'jobber-svn.pl';
my $PROG_VERSION = '42';
my $AUTHOR_NAME = 'Eric Sunshine';
my $AUTHOR_EMAIL = 'sunshine@sunshineco.com';
my $COPYRIGHT = "Copyright (C) 2000-2012 by $AUTHOR_NAME <$AUTHOR_EMAIL>\nConverted for SVN support by Marten Svanfeldt";

my $ARCHIVER_BZIP2 = {
    'name'      => 'bzip2',
    'extension' => 'tar.bz2',
    'command'   => 'tar --create --file=- ~S | bzip2 > ~D' };
my $ARCHIVER_GZIP = {
    'name'      => 'gzip',
    'extension' => 'tgz',
    'command'   => 'tar --create --file=- ~S | gzip > ~D' };
my $ARCHIVER_ZIP = {
    'name'      => 'zip',
    'extension' => 'zip',
    'command'   => 'zip -q -r ~D ~S' };
my $ARCHIVER_LZMA = {
    'name'      => 'lzma',
    'extension' => 'tar.lzma',
    'command'   => 'tar --create --file=- ~S | lzma > ~D' };

my $jobber_project_root = undef;
my $jobber_svn_base_url = undef;
my $jobber_svn_flags = '';
my $jobber_svn_user = '';
my $jobber_browseable_dir = undef;
my $jobber_package_dir = undef;
my $jobber_public_group = undef;
my $jobber_public_mode = undef;
my $jobber_temp_dir = '/tmp';
my @jobber_tasks = ();
my @jobber_archivers = ($ARCHIVER_BZIP2, $ARCHIVER_GZIP, $ARCHIVER_ZIP, $ARCHIVER_LZMA);
my %jobber_properties = ();

# SVN binary name
my $jobber_svn_command = '/usr/bin/svn';

my $CONFIG_FILE = undef;
my $TESTING = undef;
my $EXPORT = 1;
my $CONV_DIR = undef;
my $CAPTURED_OUTPUT = '';

my @SCRIPT_OPTIONS = (
    'set=s'     => \%jobber_properties,
    'config=s'  => \$CONFIG_FILE,
    'test!'     => \$TESTING,
    'export!'   => \$EXPORT,
    'help'      => \&option_help,
    'version|V' => \&option_version,
    '<>'        => \&option_error
);

my @NEW_DIRECTORIES = ();
my @NEW_FILES = ();
my @OUTDATED_FILES = ();

my @CONFIG_FILES = ('jobber.cfg', '.jobber');
my @CONFIG_DIRS = ('.');
push @CONFIG_DIRS, $ENV{'HOME'} if exists $ENV{'HOME'};

#------------------------------------------------------------------------------
# Terminate abnormally and print a textual representation of "errno".
#------------------------------------------------------------------------------
sub expire {
    my ($msg, $err) = @_;
    $err = $! unless $err;
    print STDERR "FATAL: $msg failed: $err\n";
    dump_captured();
    my $dir = conversion_dir();
    destroy_transient($dir) if $dir;
    croak 'Stopped';
}

#------------------------------------------------------------------------------
# Configuration file version assertion. The configuration file can invoke this
# function to ensure that the running jobber-svn.pl script is sufficiently
# recent. If it is not, then the script aborts. (We use 'die' rather than
# expire() because this error will be trapped by 'eval' in load_config().
#------------------------------------------------------------------------------
sub jobber_require {
    my $ver = shift;
    die "minimum version assertion failed: requested $ver, got $PROG_VERSION"
        unless $ver <= $PROG_VERSION;
}

#------------------------------------------------------------------------------
# Convert a list of pathnames into a string which can be passed to a shell
# command via the command-line.  Also protect special characters from the
# shell by escaping them.
#------------------------------------------------------------------------------
sub prepare_pathnames {
    my ($path, @paths);
    foreach $path (@_) {
	$path =~ s/ /\\ /g;
	push(@paths, $path);
    }
    return join(' ', @paths);
}


#------------------------------------------------------------------------------
# Change group ownership on a list of directories.
#------------------------------------------------------------------------------
sub change_group {
    my $group = shift;
    my @dirs = @_;
    return unless $group && @dirs;
    my $gid = getgrnam($group) or expire("getgrnam($group)");
    chown(-1, $gid, @dirs) or expire("chown(-1, $gid, ".join(' ', @dirs).')');
    if (defined($jobber_public_mode)) {
	chmod($jobber_public_mode, @dirs) or
	    expire("chmod($jobber_public_mode, ".join(' ', @dirs).')');
    }
}


#------------------------------------------------------------------------------
# Change group ownership of directories and all subdirectories (recursive).
#------------------------------------------------------------------------------
sub change_group_deep {
    my $group = shift;
    my @dirs = @_;
    my @children = ();
    find(sub{my $n=$File::Find::name; push(@children, $n) if -d $n}, @dirs);
    change_group($group, @dirs, @children);
}

#------------------------------------------------------------------------------
# Create a directory.
#------------------------------------------------------------------------------
sub create_directory {
    my ($dir, $group) = @_;
    mkdir($dir, 0755) or expire("mkdir($dir)");
    change_group($group, $dir);
}

#------------------------------------------------------------------------------
# Create a directory and all intermediate directories.
#------------------------------------------------------------------------------
sub create_directory_deep {
    my ($dir, $group) = @_;
    my @dirs = mkpath($dir);
    change_group($group, @dirs);
}

#------------------------------------------------------------------------------
# Change the working directory.
#------------------------------------------------------------------------------
sub change_directory {
    my $dir = shift;
    chdir($dir) or expire("chdir($dir)");
}

#------------------------------------------------------------------------------
# Copy a file.
#------------------------------------------------------------------------------
sub copy_file {
    my ($src, $dst) = @_;
    copy($src, $dst) or expire("copy($src,$dst)");
}

#------------------------------------------------------------------------------
# Remove a file.
#------------------------------------------------------------------------------
sub remove_file {
    my $file = shift;
    unlink($file) or expire("unlink($file)");
}

#------------------------------------------------------------------------------
# Rename a file.
#------------------------------------------------------------------------------
sub rename_file {
    my ($src, $dst) = @_;
    rename($src, $dst) or expire("rename($src,$dst)");
}

#------------------------------------------------------------------------------
# Remove a directory.
#------------------------------------------------------------------------------
sub remove_dir {
    my $dir = shift;
    rmdir($dir) or expire("rmdir($dir)");
}

#------------------------------------------------------------------------------
# Generate a temporary name in a directory.  Perl tmpnam() only works with
# '/tmp', so must do this manually, instead.
#------------------------------------------------------------------------------
sub temporary_name {
    my ($dir, $prefix, $suffix) = @_;
    $prefix = 'tmp' unless $prefix;
    $suffix = '' unless $suffix;
    my ($i, $limit) = (0, 100);
    $i++ while -e "$dir/$prefix$i$suffix" && $i < $limit;
    expire("temporary_name($dir,$prefix,$suffix)", "exceeded retry limit")
        if $i >= $limit;
    return "$dir/$prefix$i$suffix";
}

#------------------------------------------------------------------------------
# Return the name of the conversion directory. If not already set, then first
# compute it.
#------------------------------------------------------------------------------
sub conversion_dir {
  $CONV_DIR = temporary_name($jobber_temp_dir) unless $CONV_DIR;
  return $CONV_DIR;
}

#------------------------------------------------------------------------------
# Run an external shell command.
#------------------------------------------------------------------------------
sub run_command {
    my $cmd = shift;
    my $output = `( $cmd ) 2>&1`;
    $CAPTURED_OUTPUT .= "==> $cmd\n$output\n";
    expire("run_command($cmd)") if $?;
    return $output;
}

#------------------------------------------------------------------------------
# Perform a recursive scan of a directory and return a sorted list of all
# files and directories contained therein, except for the ".svn" directory and
# its control files.  Also ignores ".cvsignore" files and ".svn" directories.
#------------------------------------------------------------------------------
sub scandir {
    my $dir = shift;
    my @files;
    find(sub{my $n=$File::Find::name; $n=~s|$dir/?||; push(@files,$n) if $n},
	 $dir);
    return sort(grep(!/.cvsignore/,grep(!/.svn/,@files)));
}

#------------------------------------------------------------------------------
# Invokes the SVN `delete` command on a number of files.
#------------------------------------------------------------------------------
sub svn_remove {
    my $files = shift;
    return unless @{$files};
    my $paths = prepare_pathnames(@{$files});
    print "Invoking SVN delete: ${\scalar(@{$files})} paths\n";
    run_command("$jobber_svn_command delete $paths $jobber_svn_flags") unless $TESTING;
}

#------------------------------------------------------------------------------
# Queue an entry for removal from the SVN repository if it does not exist in
# the newly generated directory hierarchy.
#------------------------------------------------------------------------------
sub svn_queue_remove {
    my $dst = shift;
    my $file = basename($dst);
    if (-d $dst) {
	print "Pruning directory: $file\n";
	push(@OUTDATED_FILES, $dst);
    }
    else {
	print "Removing file: $file\n";
	push(@OUTDATED_FILES, $dst);
    }
}

#------------------------------------------------------------------------------
# Apply the SVN `add' command to a batch of files.  Allows specification of
# extra SVN flags.
#------------------------------------------------------------------------------
sub svn_add {
    my ($files, $flags) = @_;
    return unless @{$files};
    my $paths = prepare_pathnames(@{$files});
    $flags = '' unless defined($flags);
    print "Invoking SVN add: ${\scalar(@{$files})} paths" .
	($flags ? " [$flags]" : '') . "\n";
    run_command("$jobber_svn_command add $flags $paths $jobber_svn_flags") unless $TESTING;
}

#------------------------------------------------------------------------------
# Queue a file or directory from the newly generated directory hierarchy for
# addition to the SVN repository. 
#------------------------------------------------------------------------------
sub svn_queue_add {
    my ($src, $dst) = @_;
    my $file = basename($dst);
    if (-d $src) {
	print "Adding directory: $file\n";
	create_directory($dst);
	push(@NEW_DIRECTORIES, $dst);
    }
    else {
	# my $isbin = is_binary($src);
	print "Adding file: $file\n";
	push(@NEW_FILES, $dst);
	copy_file($src, $dst);
    }
}

#------------------------------------------------------------------------------
# File exists in both existing SVN repository and in newly generated directory
# hierarchy.  Overwrite the existing file with the newly generated one.
# Later, at commit time, SVN will determine if any changes to the file have
# actually taken place.
#------------------------------------------------------------------------------
sub svn_examine {
    my ($src, $dst) = @_;
    if (!-d $src) {
	remove_file($dst);
	copy_file($src, $dst);
    }
}

#------------------------------------------------------------------------------
# Extract the appropriate files from the SVN repository.
#------------------------------------------------------------------------------
sub svn_checkout {
    print "URL: $jobber_svn_base_url\n";
    run_command("$jobber_svn_command co $jobber_svn_base_url $jobber_svn_flags");
}

#------------------------------------------------------------------------------
# Print a summary list of files which were modified, added, or removed.
#------------------------------------------------------------------------------
sub svn_update {
    my $message = 'Modification summary:';
    my $line = '-' x length($message);
    my %dirs = ();
    foreach my $task (@jobber_tasks) {
	if (exists $task->{'olddirs'}) {
	    $dirs{$_} = 1 foreach @{$task->{'olddirs'}};
	}
    }
    if (%dirs) {
        print "$line\n$message\n";
        my $paths = join(' ', keys(%dirs));
        my $changes = run_command("$jobber_svn_command status $paths $jobber_svn_flags");
	print $changes ? $changes : "  No files modified\n", "$line\n";
    }
}

#------------------------------------------------------------------------------
# Commit files to the SVN repository.  The 'svn' command is smart enough to
# only commit files which have actually changed.
#------------------------------------------------------------------------------
sub svn_commit_dirs {
    my ($message, @dirs) = @_;
    my $dirsAsText = "@dirs";

    my $respFileName = "commitMessage";
    open(RESPFILE, ">$respFileName");
    print RESPFILE $message;
    close(RESPFILE);

    run_command("$jobber_svn_command commit --username $jobber_svn_user -F $respFileName $dirsAsText $jobber_svn_flags")
	unless $TESTING;
    unlink($respFileName);
}

#------------------------------------------------------------------------------
# Commit repaired and generated files to the SVN repository.
#------------------------------------------------------------------------------
sub svn_commit {
    my @dirs;
    my $message;
    foreach my $task (@jobber_tasks) {
	if (exists $task->{'olddirs'}) {
	    print "Committing $task->{'name'}.\n";
	    my $msg = exists $task->{'log'} ?
	        $task->{'log'} : 'Automated file repair/generation.';
	    #svn_commit_dirs($task->{'olddirs'}, $msg);
	    push(@dirs, @{$task->{'olddirs'}});
	    $message .= "$task->{'name'}: $msg\n";
	}
    }
    svn_commit_dirs("$message", @dirs);
}

#------------------------------------------------------------------------------
# Perform all tasks by invoking the appropriate command of each task.
#------------------------------------------------------------------------------
sub run_tasks {
    foreach my $task (@jobber_tasks) {
	next unless exists($task->{'command'});
	my $does_svn = exists($task->{'olddirs'});
	my $does_export = exists($task->{'export'});
	if ($does_export && !$EXPORT && !$does_svn)
	{
	      print "Skipping: $task->{'action'} $task->{'name'}.\n";
	      next;
	}
	print "$task->{'action'} $task->{'name'}.\n";
	run_command($task->{'command'});
    }
}

#------------------------------------------------------------------------------
# Scan and compare the newly generated directory hierarchies against existing
# hierarchies from the SVN repository.  For each difference between the two
# hierarchies, apply the appropriate SVN operation, adding or removing entries
# as necessary.
#------------------------------------------------------------------------------
sub apply_diffs {
    foreach my $task (@jobber_tasks) {
	next unless exists $task->{'olddirs'} and exists $task->{'newdirs'};
	print "Applying changes to $task->{'name'}.\n";

	my @olddirs = @{$task->{'olddirs'}};
	my @newdirs = @{$task->{'newdirs'}};
	foreach my $olddir (@olddirs) {
	    my $newdir = shift @newdirs;

	    print "  Scanning  ($olddir <=> $newdir).\n";
	    my @oldfiles = scandir($olddir);
	    my @newfiles = scandir($newdir);
	
	    print "  Comparing ($olddir <=> $newdir).\n";
	    my $oldfile = shift @oldfiles;
	    my $newfile = shift @newfiles;
	
	    while (defined($oldfile) || defined($newfile)) {
		if (!defined($newfile) ||
		    (defined($oldfile) && $oldfile lt $newfile)) {
		    svn_queue_remove("$olddir/$oldfile");
		    $oldfile = shift @oldfiles;
		}
		elsif (!defined($oldfile) || $oldfile gt $newfile) {
		    svn_queue_add("$newdir/$newfile", "$olddir/$newfile");
		    $newfile = shift @newfiles;
		}
		else { # Filenames are identical.
		    svn_examine("$newdir/$newfile", "$olddir/$newfile");
		    $oldfile = shift @oldfiles;
		    $newfile = shift @newfiles;
		}
	    }
	}
    }
    svn_add(\@NEW_DIRECTORIES);
    svn_add(\@NEW_FILES);
    svn_remove(\@OUTDATED_FILES);
}

#------------------------------------------------------------------------------
# Interpolate a value into a string in place of a token.
#------------------------------------------------------------------------------
sub interpolate {
    local $_ = $_[0];
    my ($token, $value) = @_[1..2];
    s/$token/$value/g or expire("token interpolation", "$token in $_");
    $_[0] = $_;
}

#------------------------------------------------------------------------------
# Post-process a browseable directory if requested.
#------------------------------------------------------------------------------
sub postprocess_browseable {
    my ($export, $dir, $indent) = @_;
    return unless exists $export->{'browseable-postprocess'};
    $indent = '' unless defined($indent);
    print "${indent}Post-processing.\n";

    my $cmd = $export->{'browseable-postprocess'};
    interpolate($cmd, '~T', $dir);
    run_command($cmd);
}

#------------------------------------------------------------------------------
# Publish a browseable copy of the generated files.
# FIXME: Presently, only publishes first directory referenced by "newdirs" key.
#------------------------------------------------------------------------------
sub publish_browseable {
    foreach my $task (@jobber_tasks) {
	next unless exists $task->{'export'}
	    and (exists $task->{'olddirs'} or exists $task->{'newdirs'});
	if (!$EXPORT)
	{
	    print "Skipped publishing $task->{'name'}.\n";
	    next;
	}
	print "Publishing $task->{'name'}.\n";
	next if $TESTING;
	create_directory_deep($jobber_browseable_dir, $jobber_public_group)
	    unless -d $jobber_browseable_dir;

	my @srclist = exists $task->{'newdirs'} ? 
	    @{$task->{'newdirs'}} : @{$task->{'olddirs'}};
	my $src = shift @srclist;		# See FIXME above.
	my $dst = "$jobber_browseable_dir/$task->{'export'}->{'dir'}";
	my $new_dir = temporary_name($jobber_browseable_dir, 'new');
	my $old_dir = temporary_name($jobber_browseable_dir, 'old');

	print "  Preparing.\n";
	run_command("cp -r \"$src\" \"$new_dir\"");
	change_group_deep($jobber_public_group, "$new_dir");
	postprocess_browseable($task->{'export'}, $new_dir, '  ');

	print "  Installing.\n";
	rename_file($dst, $old_dir) if -e $dst;
	create_directory_deep($dst, $jobber_public_group);
	# create_directory_deep also creates the directory which we want to
	# move in, so delete that first
	remove_dir($dst);
	rename_file($new_dir, $dst);

	print "  Cleaning.\n";
	rmtree($old_dir);
    }
}

#------------------------------------------------------------------------------
# Publish an archive of the generated files.
#------------------------------------------------------------------------------
sub publish_package {
    my ($archiver, $src, $dst, $base, $indent) = @_;
    $indent = '' unless defined($indent);
    print "${indent}Archiving: $archiver->{'name'}\n";
    return if $TESTING;
    my $ext = $archiver->{'extension'};
    my $tmp_pkg = temporary_name($dst, 'pkg', ".$ext");
    my $package = "$dst/$base.$ext";
    my $cmd = $archiver->{'command'};
    interpolate($cmd, '~S', $src);
    interpolate($cmd, '~D', $tmp_pkg);
    run_command($cmd);
    rename_file($tmp_pkg, $package);
}

#------------------------------------------------------------------------------
# Publish generated directory hierarchies as archives of various formats as
# indicated by the @jobber_archivers array.  The 'appear' key in the 'export'
# dictionary of the @jobber_tasks array is used to control how the packaged
# directory appears in the archive.  For instance, although the directory
# 'out/docs/html/manual' may be packaged, the 'appear' key may indicate that it
# should appear as 'CS/docs/html/manual' in the generated package.  This
# functionality is handled by temporarily giving the target directory the
# desired name just prior to archiving.  Note that during this operation, the
# current working directory is "$CONV_DIR/$jobber_project_root", and all
# operations are performed relative to this location.
# FIXME: Presently, only publishes first directory referenced by "newdirs"
# and/or "olddirs" key.
#------------------------------------------------------------------------------
sub publish_packages {
    foreach my $task (@jobber_tasks) {
	next unless exists $task->{'export'}
	    and (exists $task->{'olddirs'} or exists $task->{'newdirs'});
	if (!$EXPORT)
	{
	    print "Skipped Packaging $task->{'name'}.\n";
	    next;
	}
	print "Packaging $task->{'name'}.\n";

	my @srclist = exists $task->{'newdirs'} ? 
	    @{$task->{'newdirs'}} : @{$task->{'olddirs'}};
	my $src = shift @srclist; # See FIXME above.
	my $export = $task->{'export'};
	my $do_appear = exists $export->{'appear'};
	my $appear = $do_appear ? $export->{'appear'} : $src;

	if ($do_appear) {
	    create_directory_deep(dirname($appear));
	    rename_file($src, $appear); # Sleight-of-hand (magic).
	}

	my $base = $export->{'name'};
	my $dst = "$jobber_package_dir/" . ($export->{'package-dir'} ?
	    $export->{'package-dir'} : $export->{'dir'});
	create_directory_deep($dst, $jobber_public_group) unless $TESTING;
	foreach my $archiver (@jobber_archivers) {
	    publish_package($archiver, $appear, $dst, $base, '  ');
	}
	rename_file($appear, $src) if $do_appear; # Unmagic.
    }
}

#------------------------------------------------------------------------------
# Create the temporary working directory.
#------------------------------------------------------------------------------
sub create_transient {
    my $dir = shift;
    print "Temporary directory: $dir\n";
    create_directory($dir);
    change_directory($dir);
}

#------------------------------------------------------------------------------
# Remove the temporary working directory.
#------------------------------------------------------------------------------
sub destroy_transient {
    my $dir = shift;
    print "Purging temporary directory.\n";
    change_directory('/');
    rmtree($dir);
}

#------------------------------------------------------------------------------
# Return a canonical representation of the current time.
#------------------------------------------------------------------------------
sub time_now {
    return gmtime() . ' UTC';
}

#------------------------------------------------------------------------------
# Perform the complete process of running tasks, committing to SVN, and
# publishing packages.
#------------------------------------------------------------------------------
sub run {
    print 'BEGIN: ', time_now(), "\n";
    my $convdir = conversion_dir();
    create_transient($convdir);
    svn_checkout();
    change_directory($jobber_project_root);
    run_tasks();
    apply_diffs();
    svn_update();
    svn_commit();
    publish_browseable();
    publish_packages();
    destroy_transient($convdir);
    print 'END: ', time_now(), "\n";
}

#------------------------------------------------------------------------------
# Dump output captured from shell commands.
#------------------------------------------------------------------------------
sub dump_captured {
    print "\nCaptured output:\n\n$CAPTURED_OUTPUT";
}

#------------------------------------------------------------------------------
# Print a validation error and terminate script.
#------------------------------------------------------------------------------
sub cfg_err {
    my $var = shift;
    expire("startup", "configuration property not initialized: $var");
}

#------------------------------------------------------------------------------
# Validate configuration information. Check that all required settings have
# been given values.
#------------------------------------------------------------------------------
sub validate_config {
    $jobber_project_root or cfg_err("\$jobber_project_root");
    $jobber_svn_base_url or cfg_err("\$jobber_svn_base_url");
    @jobber_tasks        or cfg_err("\@jobber_tasks");

    foreach my $task (@jobber_tasks) {
        if (exists $task->{'export'}) {
	    $jobber_browseable_dir or cfg_err("\$jobber_browseable_dir");
	    $jobber_package_dir    or cfg_err("\$jobber_package_dir");
	    last;
	}
    }
}

#------------------------------------------------------------------------------
# Load configuration file. If specified via an option, then load that file,
# otherwise search for one.
#------------------------------------------------------------------------------
sub load_config {
    unless ($CONFIG_FILE) {
        SEARCH: foreach my $dir (@CONFIG_DIRS) {
	    foreach my $file (@CONFIG_FILES) {
		my $path = "$dir/$file";
		if (-e $path) {
		    $CONFIG_FILE = $path;
		    last SEARCH;
		}
	    }
        }
    }
    expire("no configuration", "unable to locate configuration file")
        unless $CONFIG_FILE;

    my $content;
    print "Configuration file: $CONFIG_FILE\n\n";
    {
	local $/; # Slurp file mode.
	open my $fh, '<', $CONFIG_FILE or expire("open configuration file");
	$content = <$fh>;
	close $fh;
    }
    eval $content;
    expire("load configuration", $@) if $@;

    validate_config();
}

#------------------------------------------------------------------------------
# Display an opening banner.
#------------------------------------------------------------------------------
sub banner {
    my $stream = shift;
    $stream = \*STDOUT unless $stream;
    print $stream "\n$PROG_NAME version $PROG_VERSION\n$COPYRIGHT\n\n";
}

#------------------------------------------------------------------------------
# Display usage statement.
#------------------------------------------------------------------------------
sub print_usage {
    my $stream = shift;
    $stream = \*STDOUT unless $stream;
    banner($stream);
    print $stream <<EOT;
This program performs a series of tasks to generate and repair files, commit
modified files to the SVN repository, and publish the generated files for
browsing and download.  It should be run on a periodic basis, typically by an
automated mechanism.  The tasks which it performs are controlled by a
configuration file specified by the --config option. If --config is not used,
then it looks for a file named jobber.cfg or .jobber, first in the current
working directory, and then in the home directory (\$HOME).

Usage: $PROG_NAME [options]

Options:
    -s --set property=value
                 Assign value to an abitrary property name. The configuration
                 file can access this information via the \%jobber_properties
                 hash.
    -c --config file
                 Specify the configuration file rather than searching the
                 current working directory and the home directory for a file
                 named jobber.cfg or .jobber.
    -t --test    Process all tasks but do not actually modify the SVN
                 repository or export any files.
    --noexport
		 Process all tasks and update the SVN repository but don't
		 export any files.
    -h --help    Display this usage message.
    -V --version Display the version number of @{[basename($0)]}

EOT
}

#------------------------------------------------------------------------------
# Process command-line options.
#------------------------------------------------------------------------------
sub process_options {
    GetOptions(@SCRIPT_OPTIONS) or usage_error('');
    banner();
    print "Non-destructive testing mode enabled.\n\n" if $TESTING;
    load_config();
}

sub option_help    { print_usage(\*STDOUT); exit(0); }
sub option_version { banner(\*STDOUT); exit(0); }
sub option_error   { usage_error("Unknown option: @_\n"); }

sub usage_error {
    my $msg = shift;
    print STDERR "ERROR: $msg\n" if $msg;
    print_usage(\*STDERR);
    exit(1);
}

#------------------------------------------------------------------------------
# Run the conversion.
#------------------------------------------------------------------------------
process_options();
run();
dump_captured();
