/* JsonWriter.vala
 *
 * Copyright (C) 2009 - 2016 Jerry Casiano
 *
 * This file is part of Font Manager.
 *
 * Font Manager is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Font Manager is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Font Manager.  If not, see <http://www.gnu.org/licenses/gpl-3.0.txt>.
 *
 * Author:
 *        Jerry Casiano <JerryCasiano@gmail.com>
*/

class JsonWriter : Json.Generator {

    public bool compress { get; set; default = false; }

    public JsonWriter (Json.Node root) {
        Object(root: root);
    }

    public new bool to_file (string filepath, bool use_backup = false) {
        try {
            File file = File.new_for_path(filepath);
            FileOutputStream stream = file.replace(
                                         null,
                                         use_backup,
                                         FileCreateFlags.REPLACE_DESTINATION,
                                         null);
            File parent = file.get_parent();
            if (!parent.query_exists())
                parent.make_directory_with_parents();
            if (compress) {
                indent = 0;
                pretty = false;
                var compressor = new ZlibCompressor(ZlibCompressorFormat.ZLIB);
                var compressed_stream = new ConverterOutputStream (stream, compressor);
                to_stream(compressed_stream, null);
            } else {
                indent = 4;
                pretty = true;
                to_stream(stream, null);
            }
            return true;
        } catch (Error e) {
            warning(e.message);
            return false;
        }
    }

}

/**
 * Convenience function.
 * Equivalent to #Json.Writer.to_file()
 */
public bool write_json_file (Json.Node root,
                              string filepath,
                              bool compress = false,
                              bool backup = false) {
    var writer = new JsonWriter(root);
    writer.compress = compress;
    return writer.to_file(filepath, backup);
}

/**
 * @return      #Json.Node or #null if file failed to load
 */
public Json.Node? load_json_file (string filepath, bool compressed = false) {
    try {
        var parser = new Json.Parser();
        if (compressed) {
            var stream = File.new_for_path(filepath).read();
            var decompressor = new ZlibDecompressor(ZlibCompressorFormat.ZLIB);
            var uncompressed_stream = new ConverterInputStream(stream, decompressor);
            parser.load_from_stream(uncompressed_stream);
        } else
            parser.load_from_file(filepath);
        return parser.get_root();
    } catch (Error e) {
        warning("\nFailed to load cache file : %s\n%s : skipping...", filepath, e.message);
    }
    return null;
}
