// Copyright 2019 The Prometheus Authors
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// +build linux

package sysfs

import (
	"fmt"
	"io/ioutil"
	"path/filepath"
	"strings"
)

const (
	notAffected = "Not Affected"
	vulnerable  = "Vulnerable"
	mitigation  = "Mitigation"
)

// CPUVulnerabilities retrieves a map of vulnerability names to their mitigations.
func (fs FS) CPUVulnerabilities() ([]Vulnerability, error) {
	matches, err := filepath.Glob(fs.sys.Path("devices/system/cpu/vulnerabilities/*"))
	if err != nil {
		return nil, err
	}

	vulnerabilities := make([]Vulnerability, 0, len(matches))
	for _, match := range matches {
		name := filepath.Base(match)

		value, err := ioutil.ReadFile(match)
		if err != nil {
			return nil, err
		}

		v, err := parseVulnerability(name, string(value))
		if err != nil {
			return nil, err
		}

		vulnerabilities = append(vulnerabilities, v)
	}

	return vulnerabilities, nil
}

// Vulnerability represents a single vulnerability extracted from /sys/devices/system/cpu/vulnerabilities/
type Vulnerability struct {
	CodeName   string
	State      string
	Mitigation string
}

func parseVulnerability(name, value string) (Vulnerability, error) {
	v := Vulnerability{CodeName: name}
	value = strings.TrimSpace(value)
	if value == notAffected {
		v.State = notAffected
		return v, nil
	}

	if strings.HasPrefix(value, vulnerable) {
		v.State = vulnerable
		v.Mitigation = strings.TrimPrefix(strings.TrimPrefix(value, vulnerable), ": ")
		return v, nil
	}

	if strings.HasPrefix(value, mitigation) {
		v.State = mitigation
		v.Mitigation = strings.TrimPrefix(strings.TrimPrefix(value, mitigation), ": ")
		return v, nil
	}

	return v, fmt.Errorf("unknown vulnerability state for %s: %s", name, value)
}
