/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     coarsen_1d.c                                                   */
/*                                                                          */
/* description:  coarsening of 1 dim. hierarchical meshes;                  */
/*               file contains all routines depending on DIM == 1;          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  AI_coarse_recursive_1d: coarsens a single element of the coarsening     */
/*  patch dofs in the interior of the element are removed; dofs for higher  */
/*  order at the boundary or the coarsening patch still belong to           */
/*  the parent. Do not remove them from the mesh!!!                         */
/*  WARNING: This function is called directly from submesh.c!               */
/*--------------------------------------------------------------------------*/

int AI_coarse_recursive_1d(MESH *mesh, EL *parent)
{
  FUNCNAME("AI_coarse_recursive_1d");
  int  mark;

  INFO(0,2,"\n");
  if (!parent->child[0])  /*---  leaf element, clean element marker      ---*/
  {
    mark = parent->mark;
    parent->mark = 0;
    return(mark);
  }
  else                    /*---  no leaf element, first coarsen children ---*/
  {
    EL       *child[2];

    int  mark0 = AI_coarse_recursive_1d(mesh, parent->child[0]);
    int  mark1 = AI_coarse_recursive_1d(mesh, parent->child[1]);

    mark = MAX(mark0, mark1);

    child[0] = parent->child[0];   child[1] = parent->child[1]; 

    if (mark >= 0)         /*--- element must not be coarsend            ---*/
    {
      parent->mark = 0;
      if (child[0])
	child[0]->mark = child[1]->mark = 0;

      return(0);
    }

/*--------------------------------------------------------------------------*/
/*--- and now coarsen child[0] and child[1] into parent                  ---*/
/*--------------------------------------------------------------------------*/
  
/*--------------------------------------------------------------------------*/
/*--- hand DOFs from children to parent, and add DOF at center           ---*/
/*--------------------------------------------------------------------------*/
    if (mesh->n_dof[CENTER])
      AI_reactivate_dof(mesh, parent);

/*--------------------------------------------------------------------------*/
/*  restrict dof vectors to the parents on the patch                        */
/*--------------------------------------------------------------------------*/

    if (call_coarse_restrict_1d)
    {
      RC_LIST_EL  coarse_list[1] = {{{0}}};

      coarse_list->el_info.el = parent;
      coarse_restrict(mesh, coarse_list, 1);
    }

/*--------------------------------------------------------------------------*/
/*--- remove all DOFs of children that are not used anymore              ---*/
/*--------------------------------------------------------------------------*/

    if (mesh->n_dof[VERTEX])       /*---  midpoint of parent          ---*/
    {
      free_dof(child[1]->dof[mesh->node[VERTEX]], mesh, VERTEX, false);
    }

    if (mesh->n_dof[CENTER])       /*--- center of the children       ---*/
    {
      free_dof(child[0]->dof[mesh->node[CENTER]], mesh, CENTER, false);
      free_dof(child[1]->dof[mesh->node[CENTER]], mesh, CENTER, false);
    }

    parent->child[0] = nil;
    parent->child[1] = (EL *)AI_get_leaf_data(mesh);

    if (parent->child[1] && 
	((MESH_MEM_INFO *)mesh->mem_info)->leaf_data_info->coarsen_leaf_data)
      ((MESH_MEM_INFO *)mesh->mem_info)->
	leaf_data_info->coarsen_leaf_data(parent, child);

    free_element(child[0], mesh);
    free_element(child[1], mesh);


    mesh->n_elements--;
    mesh->n_hier_elements -= 2;
    mesh->n_vertices--;

    mark++;
    parent->mark = MIN(mark,0);

    return(parent->mark);
  }

  return(0);  /*--- statement never reached                              ---*/
}

static U_CHAR coarsen_1d(MESH *mesh)
{
  int      n_elements = mesh->n_elements;
  int      i;

  call_coarse_restrict_1d = count_coarse_restrict(mesh);

  for(i = 0; i < mesh->n_macro_el; i++)
    AI_coarse_recursive_1d(mesh, mesh->macro_els[i].el);

  return (mesh->n_elements < n_elements) ? MESH_COARSENED : 0;
}
