/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.Popups 0.1

FocusScope {
    id: edit
    property real contentHeight: childrenRect.height
    property bool dirty: false
    property alias model: mixedModel

    signal activated
    signal modelUpdated

    property TextArea activeTextArea: null
    property int activeTextAreaIndex: -1
    clip: true

    function setData(data) {
        mixedModel.clear();
        dataToModel(data);
        dirty = false;
    }

    function readData() { return modelToData() }

    onFocusChanged: {
        /* When losing focus reset the properties tracking the current
           item in the list to no selection */
        if (!focus) setCurrentTextItem(-1);
    }

    Column {
        id: list
        clip: true
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: parent.top
        anchors.margins: units.gu(1.5)

        Repeater {
            model: ListModel {
                id: mixedModel
                ListElement { type: "text"; content: "" }
            }

            delegate: Loader {
                id: loader
                width: list.width
                property int itemIndex: index

                source: (index != -1) ? (type == "image" ? "ImageDelegate.qml" : "TextDelegate.qml") : ""
                focus: index === edit.activeTextAreaIndex

                Binding { target: item; property: "source"; value: content; when: type == "image" }
                Binding { target: item; property: "text"; value: content; when: type == "text" }
                Binding { target: item; property: "popover"; value: textAreaPopup; when: type == "text" }
                Binding { target: item; property: "focus"; value: loader.focus; when: type == "text" }

                Connections {
                    target: item
                    ignoreUnknownSignals: true

                    onClicked: {
                        edit.editAtIndex(index, atTop);
                        edit.activated();
                    }

                    onDeletePrevious: {
                        syncItemToModel(item, index);
                        var location = edit.deletePreviousImage(index);
                        item.cursorPosition = location;
                    }

                    onActivateClipboard: edit.activateClipboard(type, item, index)

                    onTextChanged: {
                        // Save the text while it's changing because the user is typing
                        // but not when it's changing because it's being set from a binding
                        if (activeFocus) {
                            syncItemToModel(item, index);
                            edit.dirty = true;
                        }
                    }

                    onFocusChanged: {
                        if (type != "text") return;
                        // When leaving a text area check if it's empty and remove it from
                        // the model if it's not the only text area left
                        if (!item.focus) {
                            if (mixedModel.count > 1 &&
                                mixedModel.get(index).content.trim().length == 0) {
                                mixedModel.remove(index);
                                dirty = true;
                            }
                        } else {
                            /* Keep the current item in sync when focus is switched immediately
                               to this item as a result of the user clicking inside of it */
                            edit.setCurrentTextItem(itemIndex)
                        }
                    }
                }
            }
        }
    }

    function itemAtIndex(index) {
        for (var i = 0; i < list.children.length; i++) {
            var child = list.children[i];
            if (child.itemIndex === index) return child;
        }
        return null;
    }

    // Start editing at index, or as close as possible. -1 means at the last item.
    // If index is an image, and there is not a text edit before or after
    // (depending on the value of insertBefore), then create a temporary one.
    function editAtIndex(index, insertBefore) {
        if (index === -1) index = mixedModel.count - 1;
        if (mixedModel.get(index).type === "image") {
            if (insertBefore) {
                if (index === 0) {
                    mixedModel.insert(index, { "type": "text", "content": "" });
                    setCurrentTextItem(index);
                } else {
                    if (mixedModel.get(index - 1).type === "image") {
                        mixedModel.insert(index - 1, { "type": "text", "content": "" });
                    }
                    // If we're editing the text before an image, the editing should
                    // start at the last character before the image.
                    var item = setCurrentTextItem(index - 1);
                    item.item.cursorPosition = item.item.text.length;
                }
            } else {
                if (index === mixedModel.count - 1) {
                    mixedModel.insert(index + 1, { "type": "text", "content": "" });
                    setCurrentTextItem(index + 1);
                } else {
                    if (mixedModel.get(index + 1).type === "image") {
                        mixedModel.insert(index + 1, { "type": "text", "content": "" });
                    }
                    setCurrentTextItem(index + 1);
                }
            }
        } else {
            var item = setCurrentTextItem(index);
            item.item.cursorPosition = item.item.text.length;
        }
    }

    /* Update the properties tracking the current item in the list to the new selection.
       This automatically updates focus due to the binding in the delegate that focuses
       the item in the list which index matches list.activeTextAreaIndex. */
    function setCurrentTextItem(index) {
        if (index == -1) {
            edit.activeTextAreaIndex = -1;
            edit.activeTextArea = null;
            return
        }

        var item = itemAtIndex(index);
        if (item) {
            edit.activeTextAreaIndex = index;
            edit.activeTextArea = item.item;

            /* WORKAROUND for bug https://bugs.launchpad.net/ubuntu-ui-toolkit/+bug/1163371
            The following two lines won't be normally needed, but currently focus
            proxying inside TextArea seems to be broken, and just the binding on the delegate
            isn't enough to correctly focus the current item.
            */
            item.item.focus = true;
            if (item.item.type === "text") item.item.forceActiveFocus();
        }
        return item;
    }

    function activateClipboard(type, item, index) {
       var popupObject = PopupUtils.open(imagePopup, item);
       edit.preparePopover(type, item, index, popupObject)
       if (!popupObject.canCutOrCopy && !popupObject.canPaste) {
           PopupUtils.close(popupObject);
       }
    }

    function preparePopover(type, item, index, popupObject) {
        if (type == "text") syncItemToModel(item, index);

        popupObject.canCutOrCopy = type == "image" || item.selectionStart != item.selectionEnd;
        popupObject.canPaste = (type == "text" && Clipboard.data && Clipboard.data.formats.length > 0);
        popupObject.canSelectAll = (type == "text" && !(item.selectionStart == 0 && item.selectionEnd == item.text.length))
        popupObject.targetIndex = index;
        popupObject.targetItem = item;
        popupObject.target = item;
        if (type == "text") {
            popupObject.selectionStart = item.selectionStart;
            popupObject.selectionEnd = item.selectionEnd;
        }
    }

    Component {
        id: textAreaPopup
        ClipboardPopover {
            id: popover
            Component.onCompleted: {
                if (edit.activeTextArea == null) {
                    canCutOrCopy = false
                    canPaste = false
                } else {
                    edit.preparePopover("text", edit.activeTextArea, edit.activeTextAreaIndex, popover)
                }
            }

            onCut: edit.doCut(popover)
            onCopy: edit.doCopy(popover)
            onPaste: edit.doPaste(popover)
            onSelectAll: edit.doSelectAll(popover)
        }
    }

    Component {
        id: imagePopup
        ClipboardPopover {
            id: popover
            onCut: edit.doCut(popover)
            onCopy: edit.doCopy(popover)
            onPaste: edit.doPaste(popover)
            onSelectAll: edit.doSelectAll(popover)
        }
    }

    function doCut(popover) {
        Clipboard.clear();
        var item = mixedModel.get(popover.targetIndex);
        if (item.type == "text") {
            popover.targetItem.cut();
            syncItemToModel(popover.targetItem, popover.targetIndex);
        } else {
            var data = Clipboard.newData();
            data.urls = [item.content];
            Clipboard.push(data);

            /* If the removed image was between to text blocks then merge them */
            if (popover.targetIndex > 0 &&
                mixedModel.get(popover.targetIndex - 1).type == "text" &&
                popover.targetIndex + 1 < mixedModel.count &&
                mixedModel.get(popover.targetIndex + 1).type == "text") {
                mixedModel.setProperty(popover.targetIndex - 1, "content",
                                       mixedModel.get(popover.targetIndex - 1).content + "\n" +
                                       mixedModel.get(popover.targetIndex + 1).content)
                mixedModel.remove(popover.targetIndex + 1);
                mixedModel.remove(popover.targetIndex);
            } else {
                mixedModel.remove(popover.targetIndex);
            }
        }
    }

    function doCopy(popover) {
        Clipboard.clear();
        var item = mixedModel.get(popover.targetIndex);
        if (item.type == "text") {
            popover.targetItem.copy();
        } else {
            var data = Clipboard.newData();
            data.urls = [item.content];
            Clipboard.push(data);
        }
    }

    /* We can only paste inside text blocks, and the data that is pasted
       is either a text, an image, or both (in either order) */
    function doPaste(popover) {
        var start, end;
        var targetIndex = popover.targetIndex;
        var currentText = mixedModel.get(targetIndex).content;
        var lastClipboardTextLength = 0;

        /* First split the target text block if needed */
        if (popover.selectionStart != popover.selectionEnd) {
            start = currentText.substring(0, popover.selectionStart);
            end = currentText.substring(popover.selectionEnd);
        } else {
            start = currentText.substring(0, popover.targetItem.cursorPosition);
            end = currentText.substring(popover.targetItem.cursorPosition);
        }

        mixedModel.setProperty(popover.targetIndex, "content", start);
        popover.targetItem.cursorPosition = popover.selectionEnd;

        for (var i = 0; i < Clipboard.data.formats.length; i++) {
            var format = Clipboard.data.formats[i];
            switch (format) {
            case "text/uri-list":
                /* TODO: support pasting more than one image */
                var imageUrl = Clipboard.data.urls[0];

                /* Always add a new text block after pasting the image, and update the
                   target text block to be the new block. Since there can't be two text
                   blocks one after the other in the model, and we can't paste over images,
                   then we are guaranteed not to have two consecutive text blocks after
                   this paste. */
                mixedModel.insert(targetIndex + 1, { 'type': 'image', 'content': imageUrl });
                mixedModel.insert(targetIndex + 2, { 'type': 'text', 'content': '' });
                targetIndex += 2;
                break;
            case "text/plain":
                var clipboardText = Clipboard.data.text;
                lastClipboardTextLength = clipboardText.length;
                currentText = mixedModel.get(targetIndex).content;
                mixedModel.setProperty(targetIndex, "content", currentText + clipboardText);
                break;
            }
        }

        /* Paste the remaining text from splitting the initial target text block */
        if (mixedModel.get(targetIndex).type == "text") {
            currentText = mixedModel.get(targetIndex).content;
            mixedModel.setProperty(targetIndex, "content", currentText + end);
        }
    }

    function doSelectAll(popover) {
        popover.targetItem.selectAll();
    }

    function deletePreviousImage(i) {
        /* There shouldn't be two pieces of text one after the other,
           but better check just in case */
        if (i > 0 && mixedModel.get(i - 1).type == "image") {
            if (i > 1 && mixedModel.get(i - 2).type == "text") {
                /* If there was text before the deleted image then
                   merge it into the current element */
                var text = mixedModel.get(i - 2).content
                var location = text.length;
                text += "\n" + mixedModel.get(i).content;
                mixedModel.setProperty(i, "content", text);
                mixedModel.remove(i - 1);
                mixedModel.remove(i - 2);
                return location;
            } else {
                mixedModel.remove(i - 1);
            }
        }
        return 0;
    }

    function syncItemToModel(item, index) {
        mixedModel.setProperty(index, "content", item.text);
    }

    function modelToData() {
        if (mixedModel.count == 0 ||
            (mixedModel.count == 1 && mixedModel.get(0).type == "text" &&
             mixedModel.get(0).content.trim().length == 0)) {
            return "";
        } else {
            var json = '{ "elements" : [';
            for (var i = 0; i < mixedModel.count; i++) {
                json += (i == 0 ? "" : ",") + JSON.stringify(mixedModel.get(i));
            }
            json += "]}"
            return json;
        }
    }

    function dataToModel(noteData) {
        try {
            var data = JSON.parse(noteData);
            // TODO: make more robust by checking if it's an array and if the
            // TODO: elements have the right properties.
            if (data['elements']) {
                for (var i = 0; i < data['elements'].length; i++) {
                    var element = data['elements'][i];
                    mixedModel.append(element);
                }
            }
        } catch (exception) {
            mixedModel.append({ type: 'text', content: noteData });
        }
        modelUpdated()
    }
}
