% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/touchDirectory.R
\name{touchDirectory}
\alias{touchDirectory}
\title{Touch a versioned directory}
\usage{
touchDirectory(path, date = Sys.Date(), force = FALSE)
}
\arguments{
\item{path}{String containing the path to a versioned directory.
The \code{dirname} should be the package cache while the \code{basename} should be a version number.}

\item{date}{A \link{Date} object containing the current date.
Only provided for testing.}

\item{force}{Logical scalar indicating whether to forcibly update the access date for \code{path}.}
}
\value{
The \code{<version>_dir.expiry} stub file within \code{path} is updated/created with the current date.
A \code{NULL} is invisibly returned.
}
\description{
Touch a versioned directory to indicate that it has been successfully accessed in the recent past.
}
\details{
This function should be called \emph{after} any successful access to the contents of a versioned directory,
to indicate that said directory is still in use by expiry-aware processes.
A stub file is updated with the last access time to allow \code{\link{clearDirectories}} to accurately check for staleness.

For a given \code{path} and \code{version}, this function only modifies the files on its first call.
All subsequent calls with the same two arguments, in the same R session and on the same day will have no effect.
This avoids unnecessary touching of the file system during routine use.

The caller should lock the target directory with \code{\link{lockDirectory}} before calling this function.
This ensures that another process calling \code{\link{clearDirectories}} does not delete this directory while its access time is being updated.
If the target directory is locked, any writes to the stub file itself are thread-safe, even for shared locks.

By default, this function will remember the values of \code{path} that were passed in previous calls,
and will avoid re-updating those same \code{path}s with the same date when called on the same day.
This avoids unnecessary file system writes and locks when this function is repeatedly called.
Advanced users can force an update by setting \code{force=TRUE}.
}
\examples{
# Creating the package cache.
cache.dir <- tempfile(pattern="expired_demo")
dir.create(cache.dir)

# Creating the versioned subdirectory.
version <- package_version("1.11.0")
version.dir <- file.path(cache.dir, version)
lck <- lockDirectory(version.dir)
dir.create(version.dir)

# Setting the last access time.
touchDirectory(version.dir)
list.files(cache.dir)
readLines(file.path(cache.dir, "1.11.0_dir.expiry"))

# Making sure we unlock it afterwards.
unlockDirectory(lck)

}
\seealso{
\code{\link{lockDirectory}}, which should always be called before this function.
}
\author{
Aaron Lun
}
