##
##  Copyright (c) 2002 Ivo van der Wijk, Amaze Internet Services (ivo@amaze.nl)
## 
##  All rights reserved. 
## 
##  Redistribution and use in source and binary forms, with or without 
##  modification, are permitted provided that the following conditions
##  are met:
## 
## 1. Redistributions of source code must retain the above copyright
##    notice, this list of conditions and the following disclaimer.
## 2. Redistributions in binary form must reproduce the above copyright
##    notice, this list of conditions and the following disclaimer in the
##    documentation and/or other materials provided with the distribution.
## 3. The name of the author may not be used to endorse or promote products
##    derived from this software without specific prior written permission. 
##   
##   
## THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
## IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
## OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
## IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
## INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
## NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
## DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
## THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
## (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
## THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
##

from zLOG import LOG, WARNING, BLATHER, INFO

from OFS.ObjectManager import ObjectManager
from Acquisition import aq_parent, aq_chain, aq_inner
import QuotaFolder

##
## NOTE TO DEVELOPERS:
##
## Refreshing this product may not always work. If you do not get the expected
## behaviour after refreshing, try restarting your zopeserver!
##

##
## Idea: use *args / **args for wrapper methods and apply parameters to methods
##
## QA stands for Quota Aware, QU for Quota Unaware

##
## Invoke chain of _qf_check's, so all parents can update? (for delete of
## folders etc)

def QA_setOb(self, id, object):
    ##
    ## Is it smarter to patch setObject instead? Is setOb ever invoked directly?

    ##
    ## We rely on the transaction being rolled back if an exception occurs.
    ## This means that QuotaExceeded exceptions shouldn't occur. Unfortunately,
    ## sometimes they are caught.
    if isinstance(object, ObjectManager):
        make_quota_aware(object)

    if hasattr(object, "_qa_sync"):
       object._qa_sync()
    _files, _size = QA_determine_usage(object)
    try:
        self._qf_check(_files, _size)
    except QuotaFolder.QuotaExceededException:
        ##
        ## This horrible fix is because manage_addUserFolder ignores all
        ## exceptions, and assumes there already is a UserFolder, while
        ## _setObject happily changes _objects as well while ignoring 
        ## anything that may go wrong
        # print "O", self._objects
        o = []
        ##
        ## Could this go wrong if there already is an object with id? 
        ## (replacable?) (check using getattr?)
        if not hasattr(self, id):
            for i in self._objects:
                if i['id'] != id:
                    o.append(i)
            self._objects = tuple(o)
        raise
    result = self._qu_setOb(id, object)
    return result

def QA_delOb(self, id):
    # we can't use _getOb because it doesn't allow _attrs
    ##
    ## this works with BtreeFolders as well (well, with non-basicBtreeFolders)
    obj = getattr(self, id)
    try:
        _files,_size = QA_determine_usage(obj)
    except:
        LOG('QuotaProduct', BLATHER, "Can't get size for %s" % id)
        _files, _size = 1,0

    self._qf_check(-_files, -_size) 

    return self._qu_delOb(id)

def getid(id):
    """ sometimes id is a string, sometimes a method """
    if callable(id): return id()
    return id

def findparent(object):
    """ find the direct parent """

    for parent in aq_chain(object):
        if parent is object:
            continue
        if parent == object:
            continue
        if isinstance(parent, ObjectManager):
            return parent
    return None    
    
def QA_check_quota(self, filechange, sizechange=0, objsize=0):
    """ 
        Notify a change in usage:

        filechange      number of added/removed files
        sizechange      change in size of usage
        objsize         total size of object

        changes can be negative as well
    """
    
    if hasattr(self, "_check_quota"):
        ## check hook for QuotaFolder
        self._check_quota(filechange, sizechange, objsize)
    else: 
        ## do own administration
        make_quota_aware(self)
        self._quota_filecount = self._quota_filecount + filechange
        self._quota_size = self._quota_size + sizechange

    ##
    ## Acquisition is scary. The chain looks different when traversing
    ## through ftp/webdav. basically, 'self' may appear in the chain multiple
    ## times (with A NullResource in between), and may not always be equal
    ## through 'is' (so try == as well)
    ##
    ## Tip: use aq_base(parent) XXX

    for parent in aq_chain(self):
        if parent is self:
            continue
        if parent == self:
            continue
        if hasattr(parent, "_qf_check"):
            if hasattr(parent, getid(self.id)):
                parent._qf_check(filechange,sizechange,objsize)
            else:
                for name,ob in parent.objectItems():
                    if ob.id == getid(self.id):
                        parent._qf_check(filechange,sizechange,objsize)
                        return # !!
            break
    
def QA_manage_importObject(self, file, REQUEST=None, set_owner=1):
    """ currently a useless hook. To be removed if not used """
    result = apply(self._qu_manage_importObject, (file, REQUEST, set_owner))
    return result
    
def QA_sync(self):
    """ synchronize, i.e. recursively recalculate sizes """
    self._quota_filecount = 0 # not 1, get_usage adds 1
    self._quota_size = 0

    for child in self.objectValues():
        if hasattr(child, "_qa_sync"):
            child._qa_sync()

        _files, _size = QA_determine_usage(child)
        self._quota_filecount = self._quota_filecount + _files
        self._quota_size = self._quota_size + _size

def make_quota_aware(object):
    if not hasattr(object, "_quota_filecount"):
        object._quota_filecount = 0
    if not hasattr(object, "_quota_size"):
        object._quota_size = 0

def QA_get_usage(self):
    """
        Return the size of an ObjectManager object. This is 1 + the number
        of subobjects. The size of an empty ObjectManager is considered to
        be 0
    """
    return self._quota_filecount+1, self._quota_size

def QA_determine_usage(object):
    """ 
        attempt to determine number of size and number of subobjects of object 
    """
    ##
    ## ZPT doesn't return the size of the unrendered content it seems, 
    ## which leads to
    ## very inconsistent sizes. Let's hope this fixes it...
    if zpt_installed and isinstance(object, PageTemplate):
        # workaround for ZPT
        if hasattr(object, "_text"):
            return 1, len(object._text)
        return 1, 0
    if hasattr(object, "_get_usage"):
        return object._get_usage()
    elif hasattr(object, "get_size"):
        ##
        ## Some products (such as PhotoFolders Photo) raise an exception
        ## when requested their size at init time
        try:
            _size = object.get_size()
        except: # can be any kind of exception
            LOG('QuotaProduct', BLATHER, 'Failed to get size for %s/%s' % \
                 (getid(object.id), object.meta_type))
            _size = 0
        return 1, _size
    elif hasattr(object, "getSize"):
        ## getSize is deprecated, but if it's all there is...
        return 1, object.getSize()
    else:
        LOG('QuotaProduct', BLATHER,  "Can't get size for %s/%s" % \
            (getid(object.id), object.meta_type))
        return 1, 0

##
## Patch the PythonScript object

from Products.PythonScripts.PythonScript import PythonScript

def QA_ps_write(self, text):
    """ invoked at creation, edit and upload - what else do you want? """
    oldsize = self.get_size()
    result = self._qu_write(text)
    newsize = self.get_size()

    parent = findparent(self)

    if parent:
        if hasattr(parent, getid(self.id)):
            if hasattr(parent, "_qf_check"):
                parent._qf_check(0, newsize-oldsize, newsize)

    return result

##
## Patch DT_String, which effectively patches DTMLDocument and DTMLMethod

from DocumentTemplate.DT_String import String

def QA_dt_munge(self, source_string=None, mapping=None, **vars):
    oldsize = self.get_size()
    result = apply(self._qu_dt_munge, (source_string, mapping), vars)
    newsize = self.get_size()

    parent = findparent(self)

    if parent:
        if hasattr(parent, getid(self.id)):
            if hasattr(parent, "_qf_check"):
                parent._qf_check(0, newsize-oldsize, newsize)

    return result

##
## Patch File and Image

from OFS.Image import File, Image

def QA_file_update_data(self, data, content_type=None, size=None):
    ##
    ## This method may be invoked when self.data hasn't been initialized
    if not hasattr(self, "data"):
        oldsize = 0
    else:
        oldsize = self.get_size()
    result = apply(self._qu_update_data, (data, content_type, size))
    newsize = self.get_size()

    parent = findparent(self)

    if parent:
        if hasattr(parent, getid(self.id)):
            if hasattr(parent, "_qf_check"):
                parent._qf_check(0, newsize-oldsize, newsize)
    
    return result

## 
## Patch BTreeFolder, if available

btree_installed = 0

try:
    from Products.BTreeFolder.BTreeFolder import BTreeFolder
    LOG('QuotaProduct', INFO, "Patching BTreeFolder")
    btree_installed = 1

except ImportError:
    pass
    
##
## Patch ZopePageTemplate, if available

zpt_installed = 0

try:
    from Products.PageTemplates.ZopePageTemplate import ZopePageTemplate, PageTemplate
    LOG('QuotaProduct', INFO, "Patching ZPT")
    zpt_installed = 1
except ImportError:
    zpt_installed = 0

##
## ZPT write hook
def QA_zpt_write(self, text):
    oldsize = len(self._text)
    result = self._qu_write(text)
    newsize = len(self._text)
    
    parent = findparent(self)

    if parent:
        if hasattr(parent, getid(self.id)):
            if hasattr(parent, "_qf_check"):
                parent._qf_check(0, newsize-oldsize, newsize)
    
    return result
    
##
## Install all hooks
##
## Attempt to detect when we're refreshing and when we're restarting
if not hasattr(ObjectManager, "_quota_aware"):
    ##
    ## save 'old' methods
    LOG('QuotaProduct', INFO, 'Detected server restart')
    ObjectManager._qu_setOb = ObjectManager._setOb
    ObjectManager._qu_delOb = ObjectManager._delOb
    ObjectManager._quota_aware = 1
    ObjectManager._qu_manage_importObject = ObjectManager.manage_importObject
    PythonScript._qu_write = PythonScript.write
    String._qu_dt_munge = String.munge
    File._qu_update_data = File.update_data
    Image._qu_update_data = Image.update_data

    if btree_installed:
        BTreeFolder._qu_setOb = BTreeFolder._setOb
        BTreeFolder._qu_delOb = BTreeFolder._delOb
    if zpt_installed:
        ZopePageTemplate._qu_write = ZopePageTemplate.write
else:
    LOG('QuotaProduct', INFO, 'Detected product refresh')

##
## Install the new methods. It won't hurt if this happens during a refresh
ObjectManager._setOb = QA_setOb
ObjectManager._delOb = QA_delOb
ObjectManager.manage_importObject = QA_manage_importObject
ObjectManager._qf_check = QA_check_quota
ObjectManager._get_usage = QA_get_usage
ObjectManager._qa_sync = QA_sync
PythonScript.write = QA_ps_write
String.munge = QA_dt_munge
File.update_data = QA_file_update_data
Image.update_data = QA_file_update_data

if btree_installed:
    BTreeFolder._setOb = QA_setOb
    BTreeFolder._delOb = QA_delOb
if zpt_installed:
    ZopePageTemplate.write = QA_zpt_write

##
## finally, install the QuotaFolder Product

import QuotaFolder

def initialize(context):
    try:
        context.registerClass(
            QuotaFolder.QuotaFolder,
            meta_type='QuotaFolder',
            permission='Add QuotaFolder',
            constructors=(QuotaFolder.manage_addQuotaFolderForm,
                          QuotaFolder.manage_addQuotaFolder),
            )

    except:
        import traceback
        traceback.print_exc()
